using System;

namespace gov.va.med.VBECS.VistaLinkServer.Core
{
	/// <summary>
	/// This class represents security information associated with published 
	/// procedure on RPC server. In essence, it's a security token used to authorize 
	/// caller requesting execution of published procedure. 
	/// 
	/// The class deserializes itself from host application's XML configuration file 
	/// and provides all means to authorize calling application. 
	/// </summary>
	public class PublishedProcedureSecurityInfo
	{
		private readonly string _securityContext;
		private readonly PublishedProcedureIpAccessList _ipAccessList;

		/// <summary>
		/// Constructor accepting all information required to create class as set of parameters. 
		/// May be used by derived classes for extendibility purposes. 
		/// </summary>
		/// <param name="securityContext">
		///		Published procedure security context. 
		///		Basically, it's an arbitrary string that caller should provide to run published procedure.
		///	</param>
		/// <param name="ipAccessList">
		///		List of IP addresses that are 
		///		granted access for published procedure on RPC server.
		///	</param>
		public PublishedProcedureSecurityInfo( string securityContext, PublishedProcedureIpAccessList ipAccessList )
		{
			if( securityContext == null )
				throw( new ArgumentNullException( "securityContext" ) );

			if( ipAccessList == null )
				throw( new ArgumentNullException( "ipAccessList" ) );

			_securityContext = securityContext;
			_ipAccessList = ipAccessList;
		}

		/// <summary>
		/// Verifies security evidence supplied by caller. It will throw exception if 
		/// caller is not unauthorized to run published procedure. 
		/// Derived classes should extend this method to implement additional verification.
		/// </summary>
		/// <param name="evidence">Security evidence token supplied by caller.</param>
		public virtual void Verify( CallerSecurityEvidence evidence )
		{
			if( evidence == null )
				throw( new ArgumentNullException( "evidence" ) );

			if( !_ipAccessList.IsIpAllowedAccess( evidence.ClientIp ) )
				throw( new CallerNotAuthorizedSecurityException( SR.Exceptions.PPSecurityInfoVerificationFailedIpIsNotAllowed() ) );
		
			if( evidence.SecurityContext != _securityContext )
				throw( new CallerNotAuthorizedSecurityException( SR.Exceptions.PPSecurityInfoVerificationFailedSecurityContextMismatch() ) );
		}
	}
}
